package gui.jtransitionsystem;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.FlowLayout;
import java.awt.Paint;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Set;
import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTextArea;

import org.apache.commons.collections15.Transformer;
import org.apache.commons.collections15.functors.ChainedTransformer;


import service.ComposedTransition;
import service.ComposedState;
import service.TransitionSystem;
import edu.uci.ics.jung.algorithms.layout.FRLayout;
import edu.uci.ics.jung.graph.Graph;
import edu.uci.ics.jung.graph.SparseMultigraph;
import edu.uci.ics.jung.graph.util.EdgeType;
import edu.uci.ics.jung.visualization.VisualizationViewer;
import edu.uci.ics.jung.visualization.control.CrossoverScalingControl;
import edu.uci.ics.jung.visualization.control.DefaultModalGraphMouse;
import edu.uci.ics.jung.visualization.control.ScalingControl;
import edu.uci.ics.jung.visualization.control.ModalGraphMouse.Mode;
import edu.uci.ics.jung.visualization.decorators.ToStringLabeller;
import edu.uci.ics.jung.visualization.renderers.VertexLabelAsShapeRenderer;
import edu.uci.ics.jung.visualization.renderers.Renderer.VertexLabel.Position;

class TSComposedActionEdge {
	private String label,from,to,serviceName;
	public TSComposedActionEdge(String label,String from,String to,String serviceName) {
		this.label=label;
		this.from=from;
		this.to=to;
		this.serviceName=serviceName;
	}
	public String toString() { 
		return label+","+serviceName;
	}
	public boolean equals(String label,String from,String to) { 
		return this.label.equals(label)&&this.from.equals(from)&&this.to.equals(to);
	}
}

public class JTransitionSystemComposed{
	private Graph<String,TSComposedActionEdge> graph= new SparseMultigraph<String, TSComposedActionEdge>();
	private TransitionSystem<ComposedState, ComposedTransition> ts;
	private VisualizationViewer<String,TSComposedActionEdge> vv;
	private ComposedState currentState=null;
	private ComposedTransition lastAction=null;
	private JPanel panelTransitionSystem;
	private double repulsionMultiplier=200;
	
	public JTransitionSystemComposed(TransitionSystem<ComposedState, ComposedTransition> ts) {
		
		FRLayout<String, TSComposedActionEdge> layout = new FRLayout<String, TSComposedActionEdge>(graph);
		layout.setRepulsionMultiplier(repulsionMultiplier);
		layout.setAttractionMultiplier(2);
		
		vv = new VisualizationViewer<String,TSComposedActionEdge>(layout);
    	 
    	this.ts=ts;
		this.currentState=ts.getInitial();
		addStates();
		addActions();

		
		// Setup up a new vertex to paint transformer...
		Transformer<String,Paint> vertexPaint = new Transformer<String,Paint>() {
			public Paint transform(String i) {
				if(i.equals(currentState.getName()))
					return Color.GREEN;
				else
					return Color.CYAN;
			}
			};
			Transformer<TSComposedActionEdge,Paint> edgePaint = new Transformer<TSComposedActionEdge,Paint>() {
				public Paint transform(TSComposedActionEdge actionEdge) {
						if (lastAction!= null && actionEdge.equals(lastAction.getName(),
								              lastAction.getStateFrom().getName(),
								              lastAction.getStateTo().getName()
								             )
					       )
							return Color.RED;
						else return Color.BLACK;
				}
				};
		vv.getRenderContext().setVertexFillPaintTransformer(vertexPaint);
		vv.getRenderContext().setVertexLabelTransformer(new ToStringLabeller<String>());
		vv.getRenderContext().setEdgeLabelTransformer(new ToStringLabeller<TSComposedActionEdge>());
		vv.getRenderer().getVertexLabelRenderer().setPosition(Position.CNTR);
		vv.getRenderContext().setEdgeDrawPaintTransformer(edgePaint);
        VertexLabelAsShapeRenderer<String,TSComposedActionEdge> vlasr = new VertexLabelAsShapeRenderer<String,TSComposedActionEdge>(vv.getRenderContext());
        vv.getRenderer().setVertexLabelRenderer(vlasr);
        vv.getRenderContext().setVertexShapeTransformer(vlasr);
        vv.getRenderContext().setVertexLabelTransformer(
        		// this chains together Transformers so that the html tags
        		// are prepended to the toString method output
        		new ChainedTransformer<String,String>(new Transformer[]{
        		new ToStringLabeller<String>(),
        		new Transformer<String,String>() {
					public String transform(String input) {
						return "<html><center>"+input;
					}}}));
        vv.getRenderContext().setEdgeLabelTransformer(
        		// this chains together Transformers so that the html tags
        		// are prepended to the toString method output
        		new ChainedTransformer<TSComposedActionEdge,String>(new Transformer[]{
        		new ToStringLabeller<String>(),
        		new Transformer<String,String>() {
					public String transform(String input) {
						return "<html><center><font size=\"4\" color=\"#ff0000\">"+input;
					}}}));
        
		DefaultModalGraphMouse<String, TSComposedActionEdge> gm = new DefaultModalGraphMouse<String, TSComposedActionEdge>();
		gm.setMode(Mode.EDITING);
		vv.setGraphMouse(gm);
		
		panelTransitionSystem=new JPanel(new BorderLayout());

		JPanel panelButtons=new JPanel(new FlowLayout());

		final ScalingControl scaler = new CrossoverScalingControl();	
        JButton zoomIn = new JButton("Zoom in");
        zoomIn.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                scaler.scale(vv, 1.1f, vv.getCenter());
            }
        });
        panelButtons.add(zoomIn);

        JButton zoomOut = new JButton("Zoom out");
        zoomOut.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                scaler.scale(vv, 1/1.1f, vv.getCenter());
            }
        });
        panelButtons.add(zoomOut);


        
        JButton editingMode = new JButton("Translation Mode");
        editingMode.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
            	setModeEditing();
            }
        });
        panelButtons.add(editingMode);

        JButton pickingMode = new JButton("Picking mode");
        pickingMode.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
            	setModePicking();
            }
        });
        panelButtons.add(pickingMode);

      
        
		JPanel panelGraph=new JPanel();
		panelGraph.add(vv);
		
		JTextArea textDotArea=new JTextArea(getDotFormatString());
		textDotArea.setEditable(false);
		JScrollPane scroolPaneDotArea=new JScrollPane(textDotArea);
		JSplitPane splitPane=new JSplitPane(JSplitPane.VERTICAL_SPLIT,panelGraph,scroolPaneDotArea);
		splitPane.setDividerLocation(600);
		


		
		panelTransitionSystem.add(panelButtons,BorderLayout.NORTH);
		panelTransitionSystem.add(splitPane,BorderLayout.CENTER);
		
	}
	
	private String getDotFormatString(){
		StringBuffer result=new StringBuffer();
		result.append("digraph "+ts.getName()+ " {\n");
		Iterator<ComposedTransition> itComposedAction=ts.getAllActions().iterator();
		while(itComposedAction.hasNext()){
			ComposedTransition currentAction=itComposedAction.next();
			String currentActionName=currentAction.getName();
			String currentSourceName=currentAction.getStateFrom().getName();
			String currentTargetName=currentAction.getStateTo().getName();
			String tsName=currentAction.getService().getName();
			result.append("\t"+currentSourceName+" -> "+currentTargetName+" [label=\""+currentActionName+"\" service=\""+tsName+"\"]\n");
		}
		
		LinkedList<String> initialStatesStr=new LinkedList<String>();
		LinkedList<String> finalStatesStr=new LinkedList<String>();
		Iterator<ComposedState> itAllStates=ts.getAllStates().iterator();
		while(itAllStates.hasNext()){
			ComposedState currentState=itAllStates.next();
			if(currentState.isInitial())
				initialStatesStr.add(currentState.getName());
			if(currentState.isFinal())
				finalStatesStr.add(currentState.getName());
		}
		
		result.append("\t[initial = {");
		Iterator<String> itInitial=initialStatesStr.iterator();
		boolean firstInitialAdded=false;
		while(itInitial.hasNext()){
			if(!firstInitialAdded){
				firstInitialAdded=true;
				result.append(itInitial.next());
			}else
				result.append(","+itInitial.next());
		}
		result.append("} ]\n");

		result.append("\t[final = {");
		Iterator<String> itFinal=finalStatesStr.iterator();
		boolean firstFinalAdded=false;
		while(itFinal.hasNext()){
			if(!firstFinalAdded){
				firstFinalAdded=true;
				result.append(itFinal.next());
			}else
				result.append(","+itFinal.next());
		}
		result.append("} ]\n");
		
		
		result.append("}\n");
		return result.toString();
	}

    	
	
	public void setModeEditing(){
		DefaultModalGraphMouse<String, TSComposedActionEdge> gm = new DefaultModalGraphMouse<String, TSComposedActionEdge>();
		gm.setMode(Mode.EDITING);
		vv.setGraphMouse(gm);
	}
	public void setModePicking(){
		DefaultModalGraphMouse<String, TSComposedActionEdge> gm = new DefaultModalGraphMouse<String, TSComposedActionEdge>();
		gm.setMode(Mode.PICKING);
		vv.setGraphMouse(gm);
	}
	public ComposedState getCurrentState(){
		return currentState;
	}

	public JPanel getVisualizationViewer(){
		return panelTransitionSystem;
	}
	private void addStates(){
		Iterator<ComposedState> itState=(Iterator<ComposedState>)ts.getAllStates().iterator();
		while(itState.hasNext()){
			ComposedState state=itState.next();
			graph.addVertex(state.getName());
		}
	}
	private void addActions(){
		Iterator<ComposedTransition> itAction=(Iterator<ComposedTransition>)ts.getAllActions().iterator();
		while(itAction.hasNext()){
			ComposedTransition action=itAction.next();
			graph.addEdge(new TSComposedActionEdge(action.getName(),action.getStateFrom().getName(),action.getStateTo().getName(),action.getService().getName()),
					      action.getStateFrom().getName(),
					      action.getStateTo().getName(),EdgeType.DIRECTED);
		}
		
	}
	public Set<ComposedTransition> getPossibleAction() throws Exception{
		return ts.getActionsOf(currentState);
	}
	public void executeAction(ComposedTransition action) throws Exception{
		lastAction=action;
		currentState=(ComposedState)action.getStateTo();
	}

}
